#version 400 compatibility

/*
====================================================================================================

    Copyright (C) 2020 RRe36

    All Rights Reserved unless otherwise explicitly stated.


    By downloading this you have agreed to the license and terms of use.
    These can be found inside the included license-file
    or here: https://rre36.com/copyright-license

    Violating these terms may be penalized with actions according to the Digital Millennium
    Copyright Act (DMCA), the Information Society Directive and/or similar laws
    depending on your country.

====================================================================================================
*/

/*DRAWBUFFERS:0*/
layout(location = 0) out vec4 sceneColor;

#include "/lib/head.glsl"
#include "/lib/util/encoders.glsl"
#include "/lib/util/colorspace.glsl"

const int shadowMapResolution   = 2560;     //[512 1024 1536 2048 2560 3072 3584 4096 6144 8192 16384]
const float shadowDistance      = 128.0;

const bool shadowHardwareFiltering = true;

in vec2 coord;

flat in mat3x3 lightColor;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D colortex3;

uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform sampler2DShadow shadowtex0;
uniform sampler2DShadow shadowtex1;
uniform sampler2D shadowcolor0;

uniform int frameCounter;
uniform int isEyeInWater;

uniform ivec2 eyeBrightness;
uniform ivec2 eyeBrightnessSmooth;

uniform vec2 taaOffset;
uniform vec2 viewSize;

uniform vec3 lightvec, lightvecView;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;
uniform mat4 shadowModelView, shadowModelViewInverse;
uniform mat4 shadowProjection, shadowProjectionInverse;


/* ------ includes ------ */
#define FUTIL_TBLEND
#define FUTIL_MAT16
#include "/lib/fUtil.glsl"

#include "/lib/frag/bluenoise.glsl"
#include "/lib/frag/gradnoise.glsl"
#include "/lib/util/transforms.glsl"
#include "/lib/atmos/phase.glsl"
#include "/lib/light/warp.glsl"
#include "/lib/frag/noise.glsl"

#include "/lib/atmos/waterConst.glsl"

vec3 simple_fog(vec3 scenecolor, float d, vec3 color) {
    float dist      = max(0.0, d-32.0);
    float density   = 1.0-exp(-dist*1e-3);

    return mix(scenecolor, vec3(0.0), density);
}
vec3 waterFog(vec3 scene, float d, vec3 color) {
    float density   = max(0.0, d) * waterDensity;

    vec3 transmittance = expf(-waterAttenCoeff * density);

    const vec3 scatterCoeff = vec3(5e-2, 1e-1, 2e-1);

    vec3 scatter    = 1.0-exp(-density * scatterCoeff);
        scatter    *= max(expf(-waterAttenCoeff * density), expf(-waterAttenCoeff * tau));

    return scene * transmittance + scatter * color * rcp(pi);
}

vec3 getShadowmapPos(vec3 scenePos, const float bias) {  //shadow 2d
    vec3 pos    = scenePos + vec3(bias) * lightvec;
    float a     = length(pos);
        pos     = viewMAD(shadowModelView, pos);
        pos     = projMAD(shadowProjection, pos);
        pos.z  *= 0.2;
        pos.z  -= 0.0012*(saturate(a/256.0));

        pos.xy  = shadowmapWarp(pos.xy);

    return pos*0.5+0.5;
}
vec3 toShadowmapTexturePos(vec3 shadowPos) {
    //shadowPos.z    *= 0.2;
    shadowPos.xy    = shadowmapWarp(shadowPos.xy) * 0.5 + 0.5;

    return shadowPos;
}

vec3 getShadowcol(sampler2D tex, vec2 coord) {
    vec4 x  = texture(tex, coord);
        x.rgb = sqr(x.rgb);

    if (x.a < 0.97 && x.a > 0.93) {
        x.rgb = vec3(0.1, 0.2, 1.0);
    }

    return mix(vec3(1.0), x.rgb, x.a);
}


/* ------ refraction ------ */

vec3 refract2(vec3 I, vec3 N, vec3 NF, float eta) {     //from spectrum by zombye
    float NoI = dot(N, I);
    float k = 1.0 - eta * eta * (1.0 - NoI * NoI);
    if (k < 0.0) {
        return vec3(0.0); // Total Internal Reflection
    } else {
        float sqrtk = sqrt(k);
        vec3 R = (eta * dot(NF, I) + sqrtk) * NF - (eta * NoI + sqrtk) * N;
        return normalize(R * sqrt(abs(NoI)) + eta * I);
    }
}


/* ------ reflections ------ */

#include "/lib/light/brdf.glsl"

#include "/lib/frag/labPBR.glsl"

#include "/lib/frag/ssr.glsl"

/*
These two functions used for rough reflections are based on zombye's spectrum shaders
https://github.com/zombye/spectrum
*/

mat3 getRotationMat(vec3 x, vec3 y) {
	float cosine = dot(x, y);
	vec3 axis = cross(y, x);

	float tmp = 1.0 / dot(axis, axis);
	      tmp = tmp - tmp * cosine;
	vec3 tmpv = axis * tmp;

	return mat3(
		axis.x * tmpv.x + cosine, axis.x * tmpv.y - axis.z, axis.x * tmpv.z + axis.y,
		axis.y * tmpv.x + axis.z, axis.y * tmpv.y + cosine, axis.y * tmpv.z - axis.x,
		axis.z * tmpv.x - axis.y, axis.z * tmpv.y + axis.x, axis.z * tmpv.z + cosine
	);
}
vec3 ggxFacetDist(vec3 viewDir, float roughness, vec2 xy) {
	/*
    GGX VNDF sampling
	http://www.jcgt.org/published/0007/04/01/
    */

    viewDir     = normalize(vec3(roughness * viewDir.xy, viewDir.z));

    float clsq  = dot(viewDir.xy, viewDir.xy);
    vec3 T1     = vec3(clsq > 0.0 ? vec2(-viewDir.y, viewDir.x) * inversesqrt(clsq) : vec2(1.0, 0.0), 0.0);
    vec3 T2     = vec3(-T1.y * viewDir.z, viewDir.z * T1.x, viewDir.x * T1.y - T1.x * viewDir.y);

	float r     = sqrt(xy.x);
	float phi   = tau * xy.y;
	float t1    = r * cos(phi);
	float a     = saturate(1.0 - t1 * t1);
	float t2    = mix(sqrt(a), r * sin(phi), 0.5 + 0.5 * viewDir.z);

	vec3 normalH = t1 * T1 + t2 * T2 + sqrt(saturate(a - t2 * t2)) * viewDir;

	return normalize(vec3(roughness * normalH.xy, normalH.z));
}

void main() {
        sceneColor   = stex(colortex0);  //that macro certainly makes it neater
    vec4 tex2       = stex(colortex2);
    vec4 tex3       = stex(colortex3);

    vec4 tex1       = stex(colortex1);
    vec2 sceneLmap  = decode2x8(tex1.z);
    vec3 sceneNormal = decodeNormal(tex1.xy);
    vec3 viewNormal = normalize(mat3(gbufferModelView) * sceneNormal);
    
    //~4fps for the whole part below
    
    int matID      = decodeMatID16(tex2.x);

    float sceneDepth0   = stex(depthtex0).x;
    vec3 viewPos0       = screenToViewSpace(vec3(coord, sceneDepth0));
    vec3 scenePos0      = viewToSceneSpace(viewPos0);
    vec3 sceneDir0      = normalize(scenePos0);

    float sceneDepth1   = stex(depthtex1).x;
    vec3 viewPos1       = screenToViewSpace(vec3(coord, sceneDepth1));
    vec3 scenePos1      = viewToSceneSpace(viewPos1);
    vec3 sceneDir1      = normalize(scenePos1);

    float vDotL     = dot(normalize(viewPos1), lightvecView);
    
    bool translucent = (sceneDepth0<sceneDepth1);

    bool water      = matID == 102;

    vec3 translucentAlbedo = sqr(decode3x8(tex2.z));

    materialProperties material = materialProperties(0.0001, 0.02, false, false, mat2x3(1.0));

    if (!water) material = decodeLabBasic(decode2x8(tex2.y));

    float bluenoise     = ditherBluenoise();

    if (water){
        vec3 flatNormal     = normalize(cross(dFdx(scenePos0), dFdy(scenePos0)));
        vec3 flatViewNormal = normalize(mat3(gbufferModelView) * flatNormal);

        vec3 normalCorrected = dot(viewNormal, normalize(viewPos1)) > 0.0 ? -viewNormal : viewNormal;

        vec3 refractedDir   = refract2(normalize(viewPos1), normalCorrected, flatViewNormal, rcp(1.33));
        //vec3 refractedDir   = refract(normalize(viewPos1), normalCorrected, rcp(1.33));

        float refractedDist = distance(viewPos0, viewPos1);

        vec3 refractedPos   = viewPos1 + refractedDir * refractedDist;

        vec3 screenPos      = viewToScreenSpace(refractedPos);

        float distToEdge    = max2(abs(screenPos.xy * 2.0 - 1.0));
            distToEdge      = sqr(sstep(distToEdge, 0.7, 1.0));

        screenPos.xy    = mix(screenPos.xy, coord, distToEdge);

        //vec2 refractionDelta = coord - screenPos.xy;

        float sceneDepth    = texture(depthtex1, screenPos.xy).x;

        if (sceneDepth > sceneDepth0) {
            sceneDepth1 = sceneDepth;
            viewPos1    = screenToViewSpace(vec3(screenPos.xy, sceneDepth1));
            scenePos1   = viewToSceneSpace(viewPos1);

            sceneColor.rgb  = texture(colortex0, screenPos.xy).rgb;
        }
    }
    
    float dist0     = distance(scenePos0, gbufferModelViewInverse[3].xyz);
    float dist1     = distance(scenePos1, gbufferModelViewInverse[3].xyz);

    if (translucent && isEyeInWater==0){
        if (water) sceneColor.rgb = waterFog(sceneColor.rgb, dist1-dist0, lightColor[1]);
        else if (landMask(sceneDepth1)) sceneColor.rgb = simple_fog(sceneColor.rgb, dist1-dist0, lightColor[1]);
    }

    if (landMask(sceneDepth1) && isEyeInWater==1 && translucent) sceneColor.rgb = simple_fog(sceneColor.rgb, dist1-dist0, lightColor[1]);


    /* ------ reflections ------ */
    
    #ifdef resourcepackReflectionsEnabled

    if (water || material.roughness < 0.95){
        vec3 viewDir    = normalize(viewPos0);

        vec4 reflection = vec4(0.0);
        float skyOcclusion  = cubeSmooth(sqr(linStep(sceneLmap.y, roughReflectionsThreshold - 0.2, roughReflectionsThreshold)));

        vec3 metalAlbedo = translucentAlbedo;

        vec3 fresnel    = vec3(0.0);

        #ifdef roughReflectionsEnabled

        if (material.roughness < 0.0002 || water) {
            vec3 reflectDir = reflect(viewDir, viewNormal);

            if (dot(viewDir, viewNormal) > 0.0) viewNormal = -viewNormal;

            vec3 viewNormalSSR  = viewNormal;

            if (dot(viewDir, viewNormalSSR) > 0.0) viewNormalSSR = -viewNormalSSR;

            reflection          = screenRT(viewPos0, viewNormalSSR, bluenoise, translucent);

            fresnel   = BRDFfresnel(-viewDir, viewNormal, material, metalAlbedo);
        } else {
            float ssrAlpha  = 1.0 - sstep(material.roughness, roughSSR_maxR - 0.05, roughSSR_maxR);
            
            mat3 rot    = getRotationMat(vec3(0, 0, 1), viewNormal);
            vec3 tangentV = viewDir * rot;
            float noise = bluenoise;
            float dither = ditherGradNoiseTemporal();

            const uint steps    = roughReflectionSamples;
            const float rSteps  = 1.0 / float(steps);

            for (uint i = 0; i < steps; ++i) {
                vec2 xy         = vec2(fract((i + noise) * sqr(32.0) * phi), (i + noise) * rSteps);
                vec3 roughNrm   = rot * ggxFacetDist(-tangentV, material.roughness, xy);

                if (dot(viewDir, roughNrm) > 0.0) roughNrm = -roughNrm;

                vec3 reflectDir = reflect(viewDir, roughNrm);

                vec3 viewNormalSSR  = roughNrm;

                if (dot(viewDir, viewNormalSSR) > 0.0) viewNormalSSR = -viewNormalSSR;

                vec4 currReflection     = vec4(0.0);

                if (ssrAlpha > 0.0) currReflection = screenRT_Fast(viewPos0, viewNormalSSR, dither, translucent);
                
                reflection += currReflection;
                fresnel    += BRDFfresnel(-viewDir, roughNrm, material, metalAlbedo);
            }
            reflection *= rSteps;
            fresnel    *= rSteps;
            reflection.a *= 1.0 - linStep(material.roughness, roughReflectionsThreshold * 0.66, roughReflectionsThreshold);
        }

        #else

            vec3 reflectDir = reflect(viewDir, viewNormal);

            vec3 viewNormalSSR  = viewNormal;

            if (dot(viewDir, viewNormalSSR) > 0.0) viewNormalSSR = -viewNormalSSR;

            reflection          = screenRT(viewPos0, viewNormalSSR, bluenoise, translucent);

            reflection.a *= 1.0 - linStep(material.roughness, roughReflectionsThreshold * 0.66, roughReflectionsThreshold);

            fresnel   = BRDFfresnel(-viewDir, viewNormal, material, metalAlbedo);

        #endif
        
        //vec3 fresnel   = BRDFfresnel(-viewDir, viewNormal, material, metalAlbedo);

        if (material.conductor && material.conductorComplex) reflection.rgb *= sqrt(metalAlbedo);
        
        if (material.conductor) {
            float diffuseLum    = getLuma(sceneColor.rgb);
            float reflectionLum = getLuma(reflection.rgb * fresnel);

            float darknessComp  = saturate(-(reflectionLum - diffuseLum / pi) / max(diffuseLum, reflectionLum));
                darknessComp   *= 0.25;

            sceneColor.rgb = mix(sceneColor.rgb, reflection.rgb * fresnel + sceneColor.rgb * darknessComp, reflection.a);
        }
        else sceneColor.rgb = mix(sceneColor.rgb, reflection.rgb, vec3(reflection.a) * fresnel);

        if (water || material.conductor) {
            vec3 shadowPos  = getShadowmapPos(scenePos0, 0.08);
            float shadow    = texture(shadowtex1, shadowPos);

            #ifdef lightleakWorkaroundToggle
                shadow     *= sstep(sceneLmap.y, 0.1, 0.2);
            #endif

            //if (material.conductor) shadow *= auxMat.x;

            if (shadow > 1e-2) {
                vec3 directCol  = lightColor[0];
                if (water) sceneColor.rgb += BRDF_Beckmann(-viewDir, lightvecView, viewNormal, material) * directCol;
                //else sceneColor.rgb += BRDF(-viewDir, lightvecView, viewNormal, material, metalAlbedo) * directCol;
            }
        }

        #if DEBUG_VIEW == 6
            sceneColor.rgb = reflection.rgb;
        #endif
    }

    #else

    if (water || matID == 103){
        vec3 viewDir    = normalize(viewPos0);

        vec4 reflection = vec4(0.0);
        float skyOcclusion  = cubeSmooth(sqr(linStep(sceneLmap.y, 0.7, 0.9)));
        vec3 reflectDir = reflect(viewDir, viewNormal);

        if (dot(viewDir, viewNormal) > 0.0) viewNormal = -viewNormal;

        reflection          = screenRT(viewPos0, viewNormal, bluenoise, translucent);
        
        vec3 fresnel   = BRDFfresnel(-viewDir, viewNormal, material);
        
        sceneColor.rgb = mix(sceneColor.rgb, reflection.rgb, vec3(reflection.a) * fresnel);

        if (water || matID == 103) {
            vec3 shadowPos  = getShadowmapPos(scenePos0, 0.08);
            float shadow    = texture(shadowtex1, shadowPos);

            #ifdef lightleakWorkaroundToggle
                shadow     *= sstep(sceneLmap.y, 0.1, 0.2);
            #endif

            if (shadow > 1e-2) {
                vec3 directCol  = lightColor[0];
                sceneColor.rgb += BRDF_Beckmann(-viewDir, lightvecView, viewNormal, material) * directCol;
            }
        }

        #if DEBUG_VIEW == 6
            sceneColor.rgb = reflection.rgb;
        #endif
    }

    #endif

    sceneColor.rgb    = blendTranslucencies(sceneColor.rgb, tex3, translucentAlbedo);

    if (landMask(sceneDepth0) && isEyeInWater==0) sceneColor.rgb = simple_fog(sceneColor.rgb, dist0, lightColor[1]);

    if (isEyeInWater==1) sceneColor.rgb = waterFog(sceneColor.rgb, dist0, lightColor[1]);

    if (matID == 3) {
        if (landMask(sceneDepth0)) sceneColor.rgb = sceneColor.rgb*0.6 + vec3(0.5)*v3avg(sceneColor.rgb);
        else sceneColor.rgb = sceneColor.rgb*0.7 + vec3(0.8)*v3avg(sceneColor.rgb);
    }

    sceneColor  = makeDrawbuffer(sceneColor);
}